from math import log10, pi, sqrt # Commonly used.
import cmath as c
import hpprime as h
import math as m

'''
This code provides functions useful in an RF lab, mundane day
to day conversions, rough prop loss modeling and similar.

If you are radio-active :-), the more the merrier.
Please contribute, fix, enhance with more RF functions!

Mike Markowski, mike.ab3ap@gmail.com
Mar 2024
'''

c_mps = 299792458 # m/s, speed of light in vacuum.
kBoltz = 1.3806485279e-23 # Boltzmann's constant, units of J/K.

def main():
  '''Main program simply waits for a button to be pushed.
  '''
  h.eval('print') # Clear terminal.
  h.eval('print("\n\n                    R F   C a l c u l a t i o n s\n\n")')
  h.eval('print("Version: 25 Mar 2024\n\n")')
  h.eval('print("Bugs & suggestions to:")')
  h.eval('print("Mike Markowski, mike.ab3ap@gmail.com")')
  while True:
    mouseClear() # Ignore prior key bounces.
    h.eval('wait(0.1)') # Throttle i/o loop.
    h.eval('drawmenu("Signal", "Prop", "", "", "", "Exit")') # Main menu.
    m = mousePt()
    b = softPick(m)
    if b == 0:
      menuSignal()
    elif b == 1:
      menuProp()
    elif b == 5:
      h.eval('print') # Clear terminal.
      print('\n%18s*~- Crackle the ether! -~*' % ' ') # Clear terminal.
      h.eval('wait(1.5)')
      h.eval('print')
      break

'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
R F   C a l c u l a t i o n s
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

def sigConvert(sig, unitsIn, unitsOut):
  '''Convert input measurement to new units.

  S meters are typically used in amateur radio.  S unit levels are defined
  in dBm as S1 through S9: [-121,-115,-109,-103,-97,-91,-85,-79,-73].  Strong
  signals are indicated as S9+x dB, where x is some multiple of 10.

  Inputs:
    sig (float): measurement scalar.
    unitsIn (string): units of sig: dBm, dBuV, W, S or uV.
    unitsOut (string): output units: dBm, dBuV, W, S or uV.
  Output:
    (float): measurement scalar converted to unitsOut.
  '''

  if unitsIn == unitsOut:  return sig

  # Input units to dBm.
  if   unitsIn == 'dBm':   S_dBm = sig
  elif unitsIn == 'dBuV':  S_dBm = sig - 10*log10(50) - 90
  elif unitsIn == 'mW':    S_dBm = 10*log10(sig)
  elif unitsIn == 'S':     S_dBm = (sig - 1)*6 - 121
  elif unitsIn == 'uV':    S_dBm = 10*log10(sig**2/50) - 90

  # dBm to output units.
  if   unitsOut == 'dBm':  return S_dBm
  elif unitsOut == 'dBuV': return S_dBm + 10*log10(50) + 90
  elif unitsOut == 'W':    return 10**((S_dBm - 30)/10)
  elif unitsOut == 'S':
    if   S_dBm < -121:     return '<S1'
    elif S_dBm >= -63:     return 'S9+%d dB' % (10*((S_dBm - -73)//10))
    else:                  return 'S%d' % min(9, (1 + (S_dBm + 121)//6))
  elif unitsOut == 'V':    return sqrt(50*10**(((S_dBm - 30)/10)))

def eng(x, sigfigs=0):
  """Return x in engineering notation.  That is, mantissa and exponent
  where mantissa is between -999 and 999, and exponent a multiple of
  3.  If exponent is in [-12,12] then SI prefix is used.
  """

  if x == 0:
    return '0'

  siPrefix={-12:'p',-9:'n',-6:'µ',-3:'m',0:'',3:'k',6:'M',9:'G',12:'T'}

  # Convert x to mantissa and exponent.
  exp = int(m.floor(log10(abs(x)))) # floor() needed for exp<0.
  mant = x/10**exp
  # Round mantissa to requested number of significant figures.
  mant = round(mant, sigfigs-1) if sigfigs > 0 else mant
  # Adjust so that exponent is multiple of 3.
  mult3 = exp % 3   # How many mulitples-of-three exponent must be decreased.
  exp -= mult3    # Decrease exponent.
  mant *= 10**mult3 # Increase mantissa.
  # Create format to pretty print.
  lenMant = len(str(int(abs(mant)))) # Number of digits left of decimal pt.
  fmt = '%%.%df' % max(0, sigfigs-lenMant)
  # Convert exponent to SI prefix.
  sExp = 'e%d ' % exp if abs(exp) > 12 else ' '+siPrefix[exp]
  return (fmt % mant) + sExp

def friis(f_Hz, d_m): # Free space path loss.
  return 20*log10(4*pi*d_m*f_Hz/c_mps)

def hata(f_MHz, hM_m, hB_m, d_km, envIndex):
  err = ''
  if not (150 <= f_MHz <= 1500):
    err = 'Freq not in [150, 1500] MHz.'
  if not (1 <= hM_m <= 10):
    if err != '': err += '\n'
    err += 'Mobile antenna not in [1, 10] m.'
  if not (30 <= hB_m <= 200):
    if err != '': err += '\n'
    err += 'Base antenna not in [30, 200] m.'
  if not (1 <= d_km <= 10):
    if err != '': err += '\n'
    err += 'Tx distance not in [1, 10] km.'
  if err != '':
    h.eval('msgbox("%s")' % err)
    return -1

  env = ['Rural','Suburban','City Small','City Large'][envIndex-1]
  if env == 'City Large':
    if f_MHz <= 200:
      ah = 8.29*log10(1.54*hM_m)**2 - 1.1
    else: # XXX Paper has gap between 200 and 400 MHz!
      ah = 3.2*log10(11.75*hM_m)**2 - 4.97
  else:
    ah = (1.1*log10(f_MHz) - 0.7)*hM_m - (1.56*log10(f_MHz) - 0.8)

  if env == 'Rural':
    C = -4.78*log10(f_MHz)**2 + 18.33*log10(f_MHz) - 40.98
  elif env == 'Suburban':
    C = -2*log10(f_MHz/28)**2 - 5.4
  else:
    C = 0
  A = 69.55 + 26.16*log10(f_MHz) - 13.82*log10(hB_m) - ah
  B = 44.9 - 6.55*log10(hB_m)
  return A + B*log10(d_km) + C

def menuProp():
  '''Various models of RF path loss.
  '''
  c = h.eval('X:=0;choose(X,"Propagation","FSPL","Two Ray","Hata")')
  s = ''
  if c==1:
    # Free space path loss.
    D,Ud,F,Uf,res = h.eval('res:=input(\
      {{D,[0],               {40,30,0}},\
       {U,{"m","km"},        {76,15,0}},\
       {F,[0],               {40,30,1}},\
       {V,{"Hz","MHz","GHz"},{76,15,1}}},\
       "Freespace Loss", {"Distance:","","Frequency:",""},\
      {"","","",""},\
      {1,2,100,2},{1,2,100,2}); {D,U,F,V,res}')
    if res == 0: # User hit CANCEL.
      screenClear()
      return

    if Ud == 1: # m
      d_m = D
    else:
      d_m = 1e3*D

    if Uf == 1: # Hz
      f_Hz = F
    elif Uf == 2: # MHz
      f_Hz = 1e6*F
    else: # GHz
      f_Hz = 1e9*F

    loss_dB = 20*log10(4*pi*d_m*f_Hz/c_mps)
    x,y,delta = 51,80,15
    h.eval('textout_p("Freespace pathloss:  %.1f  dB", %d,%d)' % (loss_dB,x,y))
    toAVars('loss_dB', loss_dB)
  elif c==2: # Two Ray model.
    f,uF,d,uD,hT_m,hR_m,Glos_dBi,Grfl_dBi,R,res = h.eval('res:=input(\
      {{A,[0],               {30,15,0}},\
       {B,{"Hz","MHz","GHz"},{48,15,0}},\
       {C,[0],               {30,15,1}},\
       {D,{"m","km"},        {48,15,1}},\
       {E,[0],               {30,15,2}},\
       {F,[0],               {70,15,2}},\
       {G,[0],               {30,15,3}},\
       {H,[0],               {70,15,3}},\
       {I,[0],               {30,15,4}}},\
      "Two Ray Loss",\
      {"Freq:","","Dist:","","Hgt Tx:", "Hgt Rx:","Gain Los:","Gain Rfl:","Coeff:"},\
      {"Frequency of signal","","Distance between antennas","",\
      "m, tx antenna height", "m, rx antenna height",\
      "dBi, gain on line of sight path", "dBi, gain on reflected path",\
       "Reflection coeff, typically -1"},\
      {100,2,1,2,10,3,0,0,-1},\
      {100,2,1,2,10,3,0,0,-1}); {A,B,C,D,E,F,G,H,I,res}')
    if res == 0: # User hit CANCEL.
      screenClear()
      return

    if uF == 1:   # Hz
      f_Hz = f
    elif uF == 2: # MHz
      f_Hz = 1e6*f
    else:         # GHz
      f_Hz = 1e9*f
    if uD == 1:   # m
      d_m = d
    elif uD == 2: # km
      d_m = 1e3*d

    if (f_Hz<=0 or hT_m<=0 or hR_m<=0 or d_m<=0):
      h.eval('msgbox("Make lengths & frequency > 0")')
      return
    wl_m = c_mps/f_Hz
    Glos = 10**(Glos_dBi/10)
    Grfl = 10**(Grfl_dBi/10)
    loss_dB = twoRay(wl_m, hT_m, hR_m, d_m, R, Glos, Grfl)
    x,y = 1,160
    h.eval('textout_p("Two Ray path loss:  %.1f  dB", %d,%d)' % (loss_dB,x,y))
    toAVars('loss_dB', loss_dB)
  elif c==3:
    # Hata path loss model.
    f_MHz,d_km,hM_m,hB_m,env,res = h.eval('res:=input(\
      {{F, [0], {25,15,0}},\
       {D, [0], {65,15,0}},\
       {M, [0], {25,15,1}},\
       {B, [0], {65,15,1}},\
       {E, {"Rural","Suburban","City Small","City Large"}, {25,30,2}}},\
      "Okumura-Hata Path Loss",\
      {"Freq:","Dist:","Hgt Mob:","Hgt Base:","Env:"},\
      {"150-1500 MHz, frequency",\
       "1-10 km, distance between antennas",\
       "1-10 m, mobile antenna height",\
       "30-200 m, base antenna height",\
       "Rx environment"},\
      {1000,5,2,30,2}, {1000,5,2,30,2}); {F,D,M,B,E,res}')
    if res == 0: # User hit CANCEL.
      screenClear()
      return
    env = int(env)
    loss_dB=hata(f_MHz, hM_m, hB_m, d_km, env)
    if loss_dB == -1:
      return
    x,y = 1,120
    h.eval('textout_p("Okumura-Hata path loss:  %.1f  dB", %d,%d)'
      % (loss_dB,x,y))
    toAVars('loss_dB', loss_dB)

def menuSignal():
  '''Perform as assortment of common, simple tasks.
  '''

  c = h.eval('X:=0;choose(X,"Signal","Fresnel","Convert","Noise","VSWR","λ")')
  s = '' # Result to be printed when calculated.
  if c==1: # Fresnel zone radius.
    F,U,D,V,T,W,res = h.eval('res:=input(\
      {{F,[0],               {40,20,0}},\
       {U,{"Hz","MHz","GHz"},{63,15,0}},\
       {D,[0],               {40,20,1}},\
       {V,{"m","km"},        {63,15,1}},\
       {T,[0],               {40,20,2}},\
       {W,{"m","km"},        {63,15,2}}},\
       "First Fresnel Zone Radius",\
       {"Freq:","","D:","","dTx:",""},\
      {"Frequency","","Distance between antennas","",\
       "Distance from transmitter",""},\
      {100,2,4,2,2,2},{100,2,4,2,2,2}); {F,U,D,V,T,W,res}')
    if res == 0: # User hit CANCEL.
      screenClear()
      return

    if U == 1:   f_Hz = F     # Hz
    elif U == 2: f_Hz = 1e6*F # MHz
    else:        f_Hz = 1e9*F # GHz
    if V == 1:   D_m = D     # m
    else:        D_m = D*1e3 # km
    if W == 1:   d1_m = T     # m
    else:        d1_m = T*1e3 # km
    lambda_m = c_mps/f_Hz
    d2_m = D_m - d1_m
    r_m = sqrt(lambda_m*d1_m*d2_m/D_m)
    toAVars('r_m', r_m)
    x,y = 123,120
    h.eval('textout_p("r = %sm", %d,%d)' % (eng(r_m,3),x,y))
  if c==2: # Convert units, dBm, etc.
    V,U,res = h.eval('res:=input(\
      {{V,[0],                               {30,30,0}},\
       {U,{"dBm","mW","dBµV","µV","S-units"},{64,25,0}}},\
       "Signal Level", {"Signal:",""}, {"",""},\
      {-120,1},{-120,1}); {V,U,res}')
    if res == 0: # User hit CANCEL.
      screenClear()
      return

    units = ['dBm','mW','dBuV','uV','S'][int(U)-1]
    S_dBm    = sigConvert(V, units, 'dBm')
    S_dBuV   = sigConvert(V, units, 'dBuV')
    S_W      = sigConvert(V, units, 'W')
    S_Sunits = sigConvert(V, units, 'S')
    S_V      = sigConvert(V, units, 'V')
    x1,x2,y,delta = 40,160,60,17
    h.eval('textout_p("Equivalents:", %d,%d)'   % (           x1,y))
    h.eval('textout_p("    %.1f dBm", %d,%d)'   % (S_dBm,     x1,y+2*delta))
    h.eval('textout_p("    %.1f dBµV", %d,%d)'  % (S_dBuV,    x2,y+2*delta))
    h.eval('textout_p("    %sW", %d,%d)'        % (eng(S_W,3),x1,y+3*delta))
    h.eval('textout_p("    %sV", %d,%d)'        % (eng(S_V,3),x2,y+3*delta))
    h.eval('textout_p("    %s S-units", %d,%d)' % (S_Sunits,  x1,y+5*delta))
    toAVars('S_dBm',  S_dBm)
    toAVars('S_dBuV', S_dBuV)
    toAVars('S_W',    S_W)
    toAVars('S_V',    S_V)
    h.eval('AVars("S_Sunits"):="%s"' % S_Sunits)
  elif c==3: # Thermal noise power.
    T,U,B,V,res = h.eval('res:=input(\
      {{T,[0],               {40,20,0}},\
       {U,{"K","C","F"},     {63,15,0}},\
       {B,[0],               {40,20,1}},\
       {V,{"Hz","MHz","GHz"},{63,15,1}}},\
       "Noise Power", {"Temperature:","","Bandwidth:",""},\
      {"","","",""},\
      {288,1,1,2},{288,1,1,2}); {T,U,B,V,res}')
    if res == 0: # User hit CANCEL.
      screenClear()
      return
    # Convert input temperature to K.
    if   U == 1: T_K = T
    elif U == 2: T_K = T + 273.15
    else:        T_K = (T - 32)*5/9 + 273.15
    # Convert input frequency to Hz.
    if   V == 1: B_Hz = B
    elif V == 2: B_Hz = 1e6*B
    else:        B_Hz = 1e9*B
    if B_Hz <= 0 or T_K <= 0:
      h.eval('msgbox("Make both B,T > 0")')
      return
    n_dBm = 30 + 10*log10(kBoltz*T_K*B_Hz)
    toAVars('N_dBm', n_dBm)
    x,y = 31,90
    h.eval('textout_p("Noise Power: %.1f  dBm", %d,%d)' % (n_dBm,x,y))
  elif c==4: # VSWR.
    F,U,R,V,res = h.eval('res:=input(\
      {{F,[0],               {40,20,0}},\
       {U,{"dBm","mW","W"},  {63,15,0}},\
       {R,[0],               {40,20,1}},\
       {V,{"dBm","mW","W"},  {63,15,1}}},\
       "VSWR",\
       {"Fwd:","","Refl:",""},\
      {"Forward power","","Reflected power",""},\
      {0,1,-100,1},{0,1,-100,1}); {F,U,R,V,res}')
    if res == 0: # User hit CANCEL.
      screenClear()
      return
    if U == 1: # dBm
      fwd_W = 10**((F - 30)/10)
    elif U == 2: # mW
      fwd_W = 1e3*F
    else: # W
      fwd_W = F
    if V == 1: # dBm
      rfl_W = 10**((R - 30)/10)
    elif V == 2: # mW
      rfl_W = 1e3*R
    else: # W
      rfl_W = R
    gamma = sqrt(abs(rfl_W/fwd_W))
    vswr = (1 + gamma)/(1 - gamma)
    toAVars('vswr', vswr)
    x,y = 123,90
    h.eval('textout_p("VSWR = %.1f", %d,%d)' % (vswr,x,y))
  elif c==5: # Wavelength of frequency.
    F,U,res = h.eval('res:=input(\
      {{F,[0],               {30,30,0}},\
       {U,{"Hz","MHz","GHz"},{64,15,0}}},\
       "Carrier Frequency", {"Frequency:",""}, {"",""},\
      {100,2},{100,2}); {V,U,res}')
    if res == 0: # User hit CANCEL.
      screenClear()
      return
    # Convert frequency F to Hz.
    if   U == 1: f_Hz = F
    elif U == 2: f_Hz = 1e6*F
    else:        f_Hz = 1e9*F
    n_m = c_mps/f_Hz
    toAVars('lambda_m', n_m)
    x,y = 20,70
    h.eval('textout_p("In vacuum, λ = %sm",%d,%d)' % (eng(n_m,3),x,y))

def twoRay(wl, ht, hr, d, R=-1, Glos=1, Grfl=1):
    '''Calculate path loss between two antennas when two RF rays are
    involved: a direct LOS (line of sight) ray and a single ray reflected off
    earth surface.  Gains along each ray are different due to rays cutting
    through different points in antenna pattern.

    For details, see
    https://www.gaussianwaves.com/2019/03/two-ray-ground-reflection-model/

    Inputs:
      wl (float): m, wavelength of RF rays.
      ht (float): m, height of transmit antenna.
      hr (float): m, height of receive antenna.
      d (float[]): m, horizontal distance between antennas.
      R (float): default -1, reflection coefficient.
      Glos (float): default 1, total linear (not dBi) gain along LOS path.
      Grfl (float): default 1, total linear gain along reflected path.
    Output:
      (float[]): dBm, power of received signal for each distance d_m[].
    '''

    # Eq 1 (from URL above), ray path lengths.
    dLos = sqrt((ht - hr)**2 + d**2) # Line of sight distance.
    dRef = sqrt((ht + hr)**2 + d**2) # Reflected ray distance.
    # Eq 2, phase difference of two signals.
    phi = 2*pi*(dRef - dLos)/wl # Phase difference.
    # Eq 3, for Pt=1.
    Pr = R*sqrt(Grfl)*c.exp(-1j*phi)/dRef # 2nd term.
    Pr += sqrt(Glos)/dLos # 1st term.
    Pr = (wl/(4*pi) * abs(Pr))**2
    Pr_dB = -10*log10(Pr)
    return Pr_dB

'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
    H P   P r i m e   I / O   R o u t i n e s
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

def mouseClear():
  while h.eval('mouse(1)')>=0:
    pass # Clear event queue.

def mousePt():
  while True:
    h.eval('wait(0.1)')     # Throttle i/o loop.
    f1,f2 = h.eval('mouse') # Touch info for fingers 1 and 2.
    if len(f1) > 0:         # Got a finger touch!
      return f1             # [x,y,xOrig,yOrig,type], [x,y,xOrig,yOrig,type].

def screenClear():
  h.eval('print')  # Clear terminal.
  h.eval('rect()') # Clear graphics.

def softPick(pt):         # pt is [x, y, xOrig, yOrig, type]
  return -1 if pt[1]<220 else pt[0]//53 # Soft button is 53x20 pixels.

def toAVars(varName, val):
    cmd = 'AVars("%s"):=CAS.eval("%.11e")' % (varName, val) # 12 sig figs.
    h.eval(cmd)

'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
    R u n   P r o g r a m
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
main()
